/**
 * närma Sensor - Background Service Worker
 *
 * Responsibilities:
 * 1. Maintain configuration (server URL, identity, org context).
 * 2. Manage Session ID (narma_session_id) and Device ID.
 * 3. Track Idle State.
 * 4. Aggregate activity from content scripts.
 * 5. Send telemetry events to backend.
 */

// Configuration
const DEFAULT_CONFIG = {
    serverUrl: 'http://localhost:3000/api/snapshot',
    snapshotIntervalMinutes: 1,
    enabled: true,
    narmaOrgId: '',
    roleId: '',
    orgUnitId: ''
};

// Global State
let sessionState = {
    sessionId: null,
    lastActive: Date.now(),
    idleReason: 'active'
};

const tabActivityState = {}; // Latest state from tabs

// --- Initialization ---

chrome.runtime.onInstalled.addListener(async () => {
    console.log('närma Sensor installed/updated.');
    await initializeConfig();
    await initializeIdentity();
    ensureSession();
    setupAlarm();
});

chrome.runtime.onStartup.addListener(async () => {
    console.log('närma Sensor started.');
    await initializeConfig();
    await initializeIdentity();
    ensureSession();
    setupAlarm();
});

async function initializeIdentity() {
    const data = await chrome.storage.local.get('narma_device_pid');
    if (!data.narma_device_pid) {
        const deviceId = crypto.randomUUID();
        await chrome.storage.local.set({ narma_device_pid: deviceId });
        console.log('Generated device_pid:', deviceId);
    }
}

async function initializeConfig() {
    const data = await chrome.storage.sync.get(null);
    const newConfig = { ...DEFAULT_CONFIG, ...data };
    await chrome.storage.sync.set(newConfig);
}

function ensureSession() {
    // Rotate session if expired or missing (e.g. > 30 mins idle)
    const now = Date.now();
    if (!sessionState.sessionId || (now - sessionState.lastActive > 30 * 60 * 1000)) {
        sessionState.sessionId = crypto.randomUUID();
        console.log('New Session ID:', sessionState.sessionId);
    }
    sessionState.lastActive = now;
}

// --- Idle Tracking ---

// Set detection interval (e.g., 60 seconds)
chrome.idle.setDetectionInterval(60);

chrome.idle.onStateChanged.addListener((newState) => {
    console.log('Idle state changed:', newState);
    sessionState.idleReason = newState;

    // Log an event for this state change
    logInternalEvent('idle_state_change', { state: newState });

    // If active, update timestamp
    if (newState === 'active') {
        ensureSession();
    }
});

function logInternalEvent(action, payload) {
    // We could queue this immediately or just treat it as a background event
    // For now, let's just log it to console, but ideally this pushes to a queue to be sent
    // We'll attach it to the next snapshot cycle for simplicity.
}

// --- Alarm / Scheduling ---

async function setupAlarm() {
    const config = await chrome.storage.sync.get(['snapshotIntervalMinutes']);
    const interval = config.snapshotIntervalMinutes || DEFAULT_CONFIG.snapshotIntervalMinutes;
    chrome.alarms.create('snapshot_timer', { periodInMinutes: interval });
}

chrome.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'sync' && changes.snapshotIntervalMinutes) {
        setupAlarm();
    }
});

chrome.alarms.onAlarm.addListener(async (alarm) => {
    if (alarm.name === 'snapshot_timer') {
        await processAndSendEvents();
    }
});

// --- Message Handling (from Content Script) ---

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'CURRENT_ACTIVITY') {
        const tabId = sender.tab ? sender.tab.id : message.data.tabId;
        if (tabId) {
            tabActivityState[tabId] = {
                ...message.data,
                updatedAt: Date.now()
            };
            ensureSession(); // User is active
        }
    }
});

chrome.tabs.onRemoved.addListener((tabId) => {
    delete tabActivityState[tabId];
});

// --- Telemetry Processing ---

async function processAndSendEvents() {
    const config = await chrome.storage.sync.get(null);
    if (!config.enabled) return;

    const deviceData = await chrome.storage.local.get('narma_device_pid');
    const deviceId = deviceData.narma_device_pid;

    // Collect events
    const events = [];

    // For each tab, generate events
    for (const [tabIdStr, activity] of Object.entries(tabActivityState)) {
        const tabId = parseInt(tabIdStr, 10);

        // Skip stale data (> 5 mins old)? 
        // For "focus" stream, we might want to emit even if no actions, 
        // but let's stick to what we have: activity.recent_actions + current state

        const baseEvent = {
            narma_event_id: crypto.randomUUID(), // will be partial, individual events get new IDs
            narma_org_id: config.narmaOrgId || 'unknown',
            narma_user_pid: 'browser-user', // In real app, this comes from login
            narma_device_pid: deviceId,
            narma_session_id: sessionState.sessionId,
            narma_ingest_ts_utc: new Date().toISOString(),
            role_id: config.roleId,
            org_unit_id: config.orgUnitId,
            app: 'chrome',
            app_instance: 'chrome-extension',
            tab_id: tabId
        };

        // 1. Convert Recent Actions to Events
        if (activity.recent_actions && activity.recent_actions.length > 0) {
            activity.recent_actions.forEach((action, idx) => {
                events.push({
                    ...baseEvent,
                    narma_event_id: crypto.randomUUID(),
                    narma_event_seq: idx,
                    ts_utc: action.timestamp, // captured at source
                    event_kind: 'ui_action',
                    action: action.event_type, // click, submit
                    label: action.label,
                    element_locator: action.selector,
                    window_title: activity.page_title,
                    resource: activity.url,
                    artifact_type: activity.artifact_type,    // From content script
                    artifact_id_hash: activity.artifact_id_hash, // From content script
                    payload_json: JSON.stringify({
                        tag: action.tag_name,
                        role: action.role,
                        value: action.value,
                        checked: action.checked,
                        selected_text: action.selected_text,
                        clipboard_text: action.clipboard_text
                    })
                });
            });
            // Clear recent actions after consuming
            activity.recent_actions = [];
        }

        // 2. Emit a "focus" / heartbeat event for the current state of the tab
        events.push({
            ...baseEvent,
            narma_event_id: crypto.randomUUID(),
            ts_utc: new Date().toISOString(),
            event_kind: 'focus',
            window_title: activity.page_title,
            resource: activity.url,
            // If we have artifact info in the state, include it in the heartbeat too
            artifact_type: activity.artifact_type,
            artifact_id_hash: activity.artifact_id_hash
        });
    }

    if (events.length === 0) return;

    // Send batch
    try {
        console.log(`Sending ${events.length} events to ${config.serverUrl}`);
        await sendToBackend(config.serverUrl, { events }); // Wrap in object
    } catch (e) {
        console.error('Failed to send events', e);
    }
}

async function sendToBackend(url, payload) {
    const res = await fetch(url, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
    });
    if (!res.ok) throw new Error(res.statusText);
}
