/**
 * närma Sensor - Content Script
 * 
 * Responsibilities:
 * 1. Read metadata (URL, Title, Entity ID).
 * 2. Listen for 'click', 'submit', 'change' events.
 * 3. Extract Artifacts (Jira, PRs, etc.).
 * 4. Buffer recent actions.
 * 5. Send updates to background script.
 */

const MAX_RECENT_ACTIONS = 50;
const RECENT_ACTIONS = [];
let updateScheduled = false;

// State
let currentArtifact = { type: null, id_hash: null };

/**
 * Initialize listeners
 */
(function init() {
    document.addEventListener('click', handleEvent, true);
    document.addEventListener('submit', handleEvent, true);
    document.addEventListener('change', handleEvent, true);
    document.addEventListener('focusout', handleEvent, true); // Capture state on blur

    // Focus/Visibility stream
    document.addEventListener('visibilitychange', () => {
        if (document.visibilityState === 'visible') {
            scanForArtifacts();
            scheduleUpdate();
        }
    });

    // Clipboard events
    document.addEventListener('copy', handleClipboardEvent, true);
    document.addEventListener('cut', handleClipboardEvent, true);
    document.addEventListener('paste', handleClipboardEvent, true);

    // Selection events (debounced)
    document.addEventListener('selectionchange', debounce(handleSelectionEvent, 1000), true);

    // Initial scan
    scanForArtifacts();
    scheduleUpdate();
})();

/**
 * Handle Clipboard Events
 */
function handleClipboardEvent(event) {
    const action = {
        event_type: event.type,
        timestamp: new Date().toISOString(),
        url: window.location.href
    };

    if (event.type === 'copy' || event.type === 'cut') {
        const selection = window.getSelection().toString();
        if (selection) {
            action.clipboard_text = selection.slice(0, 500); // Capture up to 500 chars
        }
    }
    // For paste, we just log that it happened (value usually captured by input change)

    RECENT_ACTIONS.push(action);
    scheduleUpdate();
}

/**
 * Handle Selection Events
 */
function handleSelectionEvent() {
    const selection = window.getSelection().toString().trim();
    if (selection && selection.length > 5 && selection.length < 1000) {
        RECENT_ACTIONS.push({
            event_type: 'selection',
            timestamp: new Date().toISOString(),
            selected_text: selection.slice(0, 500),
            url: window.location.href
        });
        scheduleUpdate();
    }
}

/**
 * Debounce helper
 */
function debounce(func, wait) {
    let timeout;
    return function (...args) {
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(this, args), wait);
    };
}

/**
 * Scan URL/Title for potential artifacts
 */
async function scanForArtifacts() {
    const url = window.location.href;
    const title = document.title;
    let type = null;
    let id = null;

    // Heuristics
    // 1. Jira: /browse/KEY-123
    const jiraMatch = url.match(/\/browse\/([A-Z]+-\d+)/);
    if (jiraMatch) {
        type = 'Ticket';
        id = jiraMatch[1];
    }

    // 2. GitHub/GitLab PR: /pull/123 or /merge_requests/123
    if (!type) {
        const prMatch = url.match(/\/(pull|merge_requests)\/(\d+)/);
        if (prMatch) {
            type = 'PR';
            id = `${new URL(url).pathname}`; // Use full path for PR uniqueness usually, or just number if repo is known
        }
    }

    // 3. Generic "Item" from entity_id meta
    if (!type) {
        const entityId = getEntityId();
        if (entityId && entityId.length < 50 && !entityId.includes('http')) {
            // Treat as generic artifact if it looks like an ID
            type = 'Entity';
            id = entityId;
        }
    }

    if (type && id) {
        currentArtifact = {
            type,
            id_hash: await sha256(id)
        };
    } else {
        currentArtifact = { type: null, id_hash: null };
    }
}

/**
 * Extract Entity ID from page
 */
function getEntityId() {
    const meta = document.querySelector('meta[name="entity_id"]');
    if (meta && meta.content) return meta.content;
    if (document.body.dataset.entityId) return document.body.dataset.entityId;
    return null;
}

/**
 * Handle relevant DOM events
 */
function handleEvent(event) {
    if (!event.target) return;
    const target = event.target;

    // Refresh artifact info on interaction (SPA nav might have changed it)
    scanForArtifacts();

    const action = {
        event_type: event.type,
        tag_name: target.tagName,
        input_type: target.type || null,
        role: target.getAttribute('role') || getAriaRole(target),
        label: getSafeLabel(target),
        selector: getSimpleSelector(target),
        timestamp: new Date().toISOString()
    };

    if (isSearchInput(target)) {
        action.value = target.value; // Search terms are allowed
        action.is_search = true;
    }

    // Capture form values (excluding passwords)
    if (target.tagName === 'INPUT' && target.type !== 'password') {
        if (target.type === 'checkbox' || target.type === 'radio') {
            action.checked = target.checked;
        } else {
            action.value = target.value;
        }
    } else if (target.tagName === 'TEXTAREA') {
        action.value = target.value;
    } else if (target.tagName === 'SELECT') {
        action.value = target.value;
        const selectedOption = target.options[target.selectedIndex];
        if (selectedOption) {
            action.selected_text = selectedOption.text;
        }
    }

    RECENT_ACTIONS.push(action);
    if (RECENT_ACTIONS.length > MAX_RECENT_ACTIONS) RECENT_ACTIONS.shift();

    scheduleUpdate();
}

/**
 * Helper: SHA-256 Hash
 */
async function sha256(message) {
    const msgBuffer = new TextEncoder().encode(message);
    const hashBuffer = await crypto.subtle.digest('SHA-256', msgBuffer);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
}

/**
 * Check if element is likely a search input
 */
function isSearchInput(el) {
    if (el.tagName !== 'INPUT') return false;
    if (el.type === 'search') return true;
    const terms = ['search', 'query', 'keyword', 'k', 'q'];
    const candidates = [el.name, el.id, el.placeholder, el.className].filter(s => typeof s === 'string');
    return candidates.some(str => terms.some(term => str.toLowerCase().includes(term)));
}

/**
 * Get a safe, anonymized label for the element
 */
function getSafeLabel(el) {
    let text = '';

    // 1. Try ARIA Label directly
    if (el.getAttribute('aria-label')) {
        text = el.getAttribute('aria-label');
    }
    // 2. Try ARIA LabelledBy
    else if (el.getAttribute('aria-labelledby')) {
        const ids = el.getAttribute('aria-labelledby').split(' ');
        const parts = ids.map(id => {
            const labelEl = document.getElementById(id);
            return labelEl ? labelEl.innerText : '';
        });
        text = parts.join(' ');
    }
    // 3. Try standard Label tag (for="id")
    else if (el.id) {
        const labelEl = document.querySelector(`label[for="${el.id}"]`);
        if (labelEl) text = labelEl.innerText;
    }

    // 4. Try closest parent Label
    if (!text) {
        const parentLabel = el.closest('label');
        if (parentLabel) {
            // Clone and remove the input itself to get just the text? 
            // Or just take innerText, which might include the input value if it's not careful, 
            // but usually fine for a label.
            text = parentLabel.innerText;
        }
    }

    // 5. Fallback: Placeholder, Name, Title, Value (for buttons)
    if (!text) {
        text = el.getAttribute('placeholder') ||
            el.getAttribute('title') ||
            el.getAttribute('name') ||
            (el.tagName === 'BUTTON' || el.type === 'submit' ? el.innerText || el.value : '') ||
            '';
    }

    text = text.replace(/\s+/g, ' ').slice(0, 100).trim();

    // Simple redaction
    if (text.match(/[^@\s]+@[^@\s]+\.[^@\s]+/)) return '[EMAIL_REDACTED]';
    if (text.match(/[\d-\(\)\s]{10,}/)) return '[PHONE/ID_REDACTED]';
    return text;
}

/**
 * Generate a simple selector
 */
function getSimpleSelector(el) {
    let sel = el.tagName.toLowerCase();
    if (el.id) sel += `#${el.id}`;
    else if (el.classList.length > 0) sel += `.${Array.from(el.classList)[0]}`;
    return sel;
}

function getAriaRole(el) {
    if (el.tagName === 'BUTTON') return 'button';
    if (el.tagName === 'A') return 'link';
    if (el.tagName === 'INPUT') return el.type === 'checkbox' ? 'checkbox' : (el.type === 'radio' ? 'radio' : 'textbox');
    return null;
}

/**
 * Send current snapshot to background
 */
function scheduleUpdate() {
    if (updateScheduled) return;
    updateScheduled = true;
    setTimeout(() => {
        sendSnapshot();
        updateScheduled = false;
    }, 1000);
}

function sendSnapshot() {
    try {
        const payload = {
            type: 'CURRENT_ACTIVITY',
            data: {
                tabId: null, // Filled by background
                url: window.location.href,
                page_title: document.title,
                entity_id: getEntityId(),
                recent_actions: RECENT_ACTIONS,
                artifact_type: currentArtifact.type,
                artifact_id_hash: currentArtifact.id_hash
            }
        };
        chrome.runtime.sendMessage(payload).catch(() => { });
    } catch (e) { }
}
